// 使用立即执行函数避免污染全局作用域
(function() {
    'use strict';
    
    // 配置常量
    const CONFIG = {
        STORAGE_KEYS: {
            THEME: 'siyuan-quote-theme',
            SIDEBAR_VISIBLE: 'siyuan-quote-sidebar-visible',
            SIDEBAR_COLLAPSED: 'siyuan-quote-sidebar-collapsed',
            SIDEBAR_POSITION: 'siyuan-quote-sidebar-position'
        },
        HOTKEYS: {
            TOGGLE_SIDEBAR: { ctrlKey: true, shiftKey: true, key: 'T' },
            FOCUS_SEARCH: { ctrlKey: true, key: 'F' }
        },
        DEFAULT_THEME: 'default',
        DEBOUNCE_DELAY: 300
    };
    
    // 主题配置 - 28个完整主题配置
    const THEME_CONFIGS = Object.freeze({
        default: { name: '深蓝经典', desc: '优雅深蓝渐变，沉稳大气', color: 'linear-gradient(135deg, #3498db, #2c3e50)' },
        purple: { name: '神秘紫', desc: '深邃紫色，神秘优雅', color: 'linear-gradient(135deg, #9b59b6, #8e44ad)' },
        green: { name: '翡翠绿', desc: '自然生机，清新舒适', color: 'linear-gradient(135deg, #1abc9c, #16a085)' },
        rose: { name: '玫瑰金', desc: '温柔浪漫，温暖宜人', color: 'linear-gradient(135deg, #e6a4b4, #d87093)' },
        gold: { name: '暗金奢', desc: '奢华金色，高贵典雅', color: 'linear-gradient(135deg, #daa520, #b8860b)' },
        lava: { name: '熔岩红', desc: '炽热激情，活力四射', color: 'linear-gradient(135deg, #ff4500, #cc3700)' },
        ice: { name: '冰蓝光', desc: '科技冷感，清爽现代', color: 'linear-gradient(135deg, #4cc9f0, #00a8e8)' },
        star: { name: '星空紫', desc: '梦幻宇宙，深邃神秘', color: 'linear-gradient(135deg, #7d6bff, #6a5acd)' },
        twilight: { name: '暮光之城', desc: '暮色紫红，浪漫神秘', color: 'linear-gradient(135deg, #c154c1, #8a2be2)' },
        ocean: { name: '海洋深处', desc: '深邃海洋，宁静致远', color: 'linear-gradient(135deg, #1e90ff, #1049b4)' },
        amber: { name: '琥珀黄昏', desc: '温暖琥珀，复古怀旧', color: 'linear-gradient(135deg, #ffa500, #e69500)' },
        lime: { name: '青柠清新', desc: '清新青柠，活力四射', color: 'linear-gradient(135deg, #7cfc00, #6ad700)' },
        mist: { name: '迷雾灰紫', desc: '朦胧迷雾，神秘优雅', color: 'linear-gradient(135deg, #a9a9cc, #8a8ab0)' },
        vermillion: { name: '朱砂红', desc: '古典朱砂，典雅华丽', color: 'linear-gradient(135deg, #e34234, #b73023)' },
        jade: { name: '翡翠绿', desc: '东方翡翠，温润典雅', color: 'linear-gradient(135deg, #00ced1, #008b8b)' },
        void: { name: '深空黑', desc: '极简深空，现代感强', color: 'linear-gradient(135deg, #555555, #282828)' },
        coral: { name: '珊瑚橙', desc: '温暖珊瑚，活泼可爱', color: 'linear-gradient(135deg, #ff7f50, #e66e44)' },
        lavender: { name: '薰衣草紫', desc: '淡雅薰衣草，浪漫温馨', color: 'linear-gradient(135deg, #b57edc, #a569bd)' },
        mint: { name: '薄荷绿', desc: '清新薄荷，清凉舒爽', color: 'linear-gradient(135deg, #3eb489, #2e8b57)' },
        coffee: { name: '咖啡棕', desc: '香浓咖啡，温暖舒适', color: 'linear-gradient(135deg, #8b4513, #654321)' },
        sakura: { name: '樱花粉', desc: '浪漫樱花，温柔梦幻', color: 'linear-gradient(135deg, #ffb6c1, #ff69b4)' },
        steel: { name: '钢铁蓝灰', desc: '工业钢铁，冷峻硬朗', color: 'linear-gradient(135deg, #4682b4, #36648b)' },
        sunset: { name: '日落橙紫', desc: '日落西山，绚丽多彩', color: 'linear-gradient(135deg, #ff6b6b, #ff8c69)' },
        peacock: { name: '孔雀蓝绿', desc: '孔雀羽毛，华丽高贵', color: 'linear-gradient(135deg, #33a1c9, #008080)' },
        grape: { name: '葡萄紫红', desc: '成熟葡萄，浓郁奢华', color: 'linear-gradient(135deg, #8a2be2, #6a13d1)' },
        brass: { name: '黄铜复古', desc: '复古黄铜，怀旧质感', color: 'linear-gradient(135deg, #b5a642, #9c8c36)' },
        moss: { name: '苔藓绿', desc: '自然苔藓，生机盎然', color: 'linear-gradient(135deg, #8fbc8f, #2e8b57)' },
        orchid: { name: '兰花紫', desc: '优雅兰花，高贵典雅', color: 'linear-gradient(135deg, #da70d6, #ba55d3)' }
    });
    
    // 缓存DOM元素
    let domCache = {
        sidePanel: null,
        searchInput: null,
        themeList: null,
        themeOptions: null,
        randomBtn: null,
        collapseBtn: null,
        closeBtn: null,
        noResults: null,
        counter: null
    };
    
    // 状态管理
    let state = {
        isCollapsed: false,
        currentTheme: CONFIG.DEFAULT_THEME,
        isDragging: false,
        dragState: null,
        themeObserver: null,
        eventListeners: []
    };
    
    // 工具函数 - 防抖
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    // 工具函数 - 节流
    function throttle(func, limit) {
        let inThrottle;
        return function(...args) {
            if (!inThrottle) {
                func.apply(this, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    }
    
    // 工具函数 - 安全获取元素
    function getElement(selector, parent = document) {
        return parent.querySelector(selector);
    }
    
    // 工具函数 - 安全获取所有元素
    function getElements(selector, parent = document) {
        return parent.querySelectorAll(selector);
    }
    
    // 工具函数 - 存储操作
    const storage = {
        get: (key) => {
            try {
                return localStorage.getItem(key);
            } catch (e) {
                console.warn(`无法读取存储: ${key}`, e);
                return null;
            }
        },
        set: (key, value) => {
            try {
                localStorage.setItem(key, value);
            } catch (e) {
                console.warn(`无法写入存储: ${key}`, e);
            }
        },
        getJSON: (key) => {
            try {
                const value = storage.get(key);
                return value ? JSON.parse(value) : null;
            } catch (e) {
                console.warn(`无法解析JSON: ${key}`, e);
                return null;
            }
        },
        setJSON: (key, value) => {
            try {
                storage.set(key, JSON.stringify(value));
            } catch (e) {
                console.warn(`无法序列化JSON: ${key}`, e);
            }
        }
    };
    
    // 主题切换函数 - 核心修复
    function setQuoteTheme(themeName) {
        if (!THEME_CONFIGS[themeName]) {
            console.warn(`主题 "${themeName}" 不存在，使用默认主题`);
            themeName = CONFIG.DEFAULT_THEME;
        }
        
        const body = document.body;
        const oldTheme = state.currentTheme;
        
        // 移除所有主题类
        const allThemeClasses = Object.keys(THEME_CONFIGS).map(key => `theme-quote-${key}`);
        body.classList.remove(...allThemeClasses);
        
        // 添加新主题类
        body.classList.add(`theme-quote-${themeName}`);
        state.currentTheme = themeName;
        
        // 存储设置
        storage.set(CONFIG.STORAGE_KEYS.THEME, themeName);
        
        // 更新UI
        updateActiveThemeUI(themeName);
        
        console.log(`主题切换: ${oldTheme} → ${themeName} (${THEME_CONFIGS[themeName]?.name})`);
    }
    
    // 创建侧边栏
    function createSidebar() {
        if (domCache.sidePanel) return domCache.sidePanel;
        
        const sidePanel = document.createElement('div');
        sidePanel.id = 'quote-theme-sidebar';
        sidePanel.className = 'quote-theme-sidebar';
        
        const themeListHTML = generateThemeListHTML();
        const totalThemes = Object.keys(THEME_CONFIGS).length;
        
        sidePanel.innerHTML = `
            <style>
                .quote-theme-sidebar {
                    position: fixed;
                    left: 10px;
                    top: 120px;
                    background: var(--b3-theme-surface);
                    border: 1px solid var(--b3-theme-surface-lighter);
                    border-radius: 12px;
                    padding: 15px;
                    z-index: 1000;
                    box-shadow: 0 6px 20px rgba(0,0,0,0.1);
                    width: 220px;
                    backdrop-filter: blur(10px);
                    background: rgba(var(--b3-theme-surface-rgb), 0.95);
                    transition: all 0.3s ease;
                    max-height: calc(100vh - 140px);
                    overflow-y: auto;
                }
                /* 精简版CSS - 只保留必要的样式 */
                .sidebar-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; padding-bottom: 10px; border-bottom: 1px solid var(--b3-theme-surface-lighter); }
                .sidebar-title { font-weight: 600; font-size: 14px; color: var(--b3-theme-on-surface); display: flex; align-items: center; gap: 8px; }
                .sidebar-title::before { content: "🎨"; font-size: 16px; }
                .sidebar-actions { display: flex; gap: 8px; }
                .action-btn { background: none; border: none; color: var(--b3-theme-on-surface-light); cursor: pointer; font-size: 14px; width: 24px; height: 24px; display: flex; align-items: center; justify-content: center; border-radius: 4px; transition: all 0.2s; }
                .action-btn:hover { background: var(--b3-theme-surface-lighter); }
                .close-btn:hover { color: var(--b3-theme-error); }
                .collapse-btn:hover { color: var(--b3-theme-primary); }
                .theme-list { display: flex; flex-direction: column; gap: 8px; max-height: 400px; overflow-y: auto; padding-right: 5px; }
                .theme-option { display: flex; align-items: center; padding: 10px; border-radius: 8px; cursor: pointer; transition: all 0.2s; border: 2px solid transparent; background: var(--b3-theme-surface); flex-shrink: 0; }
                .theme-option:hover { background: var(--b3-theme-surface-lighter); transform: translateX(2px); }
                .theme-option.active { border-color: var(--b3-theme-primary); background: rgba(var(--b3-theme-primary-rgb), 0.08); }
                .theme-color { width: 24px; height: 24px; border-radius: 6px; margin-right: 12px; border: 2px solid rgba(0,0,0,0.1); flex-shrink: 0; }
                .theme-info { flex: 1; min-width: 0; }
                .theme-name { font-size: 13px; font-weight: 500; color: var(--b3-theme-on-surface); margin-bottom: 2px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
                .theme-desc { font-size: 11px; color: var(--b3-theme-on-surface-light); opacity: 0.7; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
                .random-section { margin-top: 15px; padding-top: 15px; border-top: 1px solid var(--b3-theme-surface-lighter); }
                .random-btn { width: 100%; background: linear-gradient(135deg, var(--b3-theme-secondary), var(--b3-theme-primary)); color: white; border: none; border-radius: 8px; padding: 10px 15px; cursor: pointer; font-size: 13px; font-weight: 500; text-align: center; transition: all 0.2s; display: flex; align-items: center; justify-content: center; gap: 6px; }
                .random-btn:hover { opacity: 0.9; transform: translateY(-1px); box-shadow: 0 4px 12px rgba(var(--b3-theme-primary-rgb), 0.3); }
                .current-indicator { font-size: 10px; color: var(--b3-theme-primary); background: rgba(var(--b3-theme-primary-rgb), 0.1); padding: 2px 6px; border-radius: 10px; margin-left: 8px; font-weight: 500; flex-shrink: 0; }
                .collapsed .theme-list, .collapsed .random-section, .collapsed .search-box, .collapsed .theme-counter { display: none; }
                .collapsed .sidebar-header { margin-bottom: 0; border-bottom: none; }
                .collapsed { width: 60px; padding: 15px 10px; }
                .collapsed .sidebar-title span { display: none; }
                .collapsed .action-btn:not(.collapse-btn) { display: none; }
                .theme-counter { font-size: 11px; color: var(--b3-theme-on-surface-light); margin-top: 5px; text-align: center; opacity: 0.7; }
                .search-box { margin-bottom: 10px; }
                .search-input { width: 100%; padding: 8px 12px; border: 1px solid var(--b3-theme-surface-lighter); border-radius: 6px; background: var(--b3-theme-surface); color: var(--b3-theme-on-surface); font-size: 12px; transition: all 0.2s; }
                .search-input:focus { outline: none; border-color: var(--b3-theme-primary); box-shadow: 0 0 0 2px rgba(var(--b3-theme-primary-rgb), 0.1); }
                .search-input::placeholder { color: var(--b3-theme-on-surface-light); }
                .no-results { text-align: center; padding: 20px; color: var(--b3-theme-on-surface-light); font-size: 12px; display: none; }
            </style>
            
            <div class="sidebar-header">
                <div class="sidebar-title">
                    <span>引述块主题</span>
                </div>
                <div class="sidebar-actions">
                    <button class="action-btn collapse-btn" title="折叠/展开">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M15 18l-6-6 6-6"/>
                        </svg>
                    </button>
                    <button class="action-btn close-btn" title="关闭">
                        ×
                    </button>
                </div>
            </div>
            
            <div class="search-box">
                <input type="text" class="search-input" placeholder="搜索主题..." />
            </div>
            
            <div class="theme-list">
                ${themeListHTML}
            </div>
            
            <div class="no-results">未找到匹配的主题</div>
            
            <div class="theme-counter">共 ${totalThemes} 个主题</div>
            
            <div class="random-section">
                <button class="random-btn">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M17 2.1l4 4-4 4"/>
                        <path d="M3 12.1v-2a4 4 0 0 1 4-4h14"/>
                        <path d="M7 21.9l-4-4 4-4"/>
                        <path d="M21 11.9v2a4 4 0 0 1-4 4H3"/>
                    </svg>
                    随机主题
                </button>
            </div>
        `;
        
        document.body.appendChild(sidePanel);
        
        // 缓存DOM元素
        cacheDOMElements(sidePanel);
        
        return sidePanel;
    }
    
    // 缓存DOM元素
    function cacheDOMElements(sidePanel) {
        domCache = {
            sidePanel,
            searchInput: getElement('.search-input', sidePanel),
            themeList: getElement('.theme-list', sidePanel),
            themeOptions: getElements('.theme-option', sidePanel),
            randomBtn: getElement('.random-btn', sidePanel),
            collapseBtn: getElement('.collapse-btn', sidePanel),
            closeBtn: getElement('.close-btn', sidePanel),
            noResults: getElement('.no-results', sidePanel),
            counter: getElement('.theme-counter', sidePanel)
        };
    }
    
    // 生成主题列表HTML
    function generateThemeListHTML() {
        const themeKeys = Object.keys(THEME_CONFIGS);
        const htmlFragments = [];
        
        themeKeys.forEach((themeKey) => {
            const theme = THEME_CONFIGS[themeKey];
            // 首次加载时没有当前指示器
            htmlFragments.push(`
                <div class="theme-option" data-theme="${themeKey}">
                    <div class="theme-color" style="background: ${theme.color};"></div>
                    <div class="theme-info">
                        <div class="theme-name">${theme.name}</div>
                        <div class="theme-desc">${theme.desc}</div>
                    </div>
                </div>
            `);
        });
        
        return htmlFragments.join('');
    }
    
    // 更新活动主题UI
    function updateActiveThemeUI(themeName) {
        const options = domCache.themeOptions;
        if (!options) return;
        
        // 批量更新，减少重排
        requestAnimationFrame(() => {
            for (let i = 0; i < options.length; i++) {
                const option = options[i];
                const optionTheme = option.dataset.theme;
                const isActive = optionTheme === themeName;
                
                // 更新active类
                option.classList.toggle('active', isActive);
                
                // 更新"当前"指示器
                const themeNameEl = option.querySelector('.theme-name');
                let indicator = themeNameEl.querySelector('.current-indicator');
                
                if (isActive && !indicator) {
                    indicator = document.createElement('span');
                    indicator.className = 'current-indicator';
                    indicator.textContent = '当前';
                    themeNameEl.appendChild(indicator);
                } else if (!isActive && indicator) {
                    indicator.remove();
                }
            }
            
            updateCounter();
        });
    }
    
    // 更新计数器
    function updateCounter() {
        if (!domCache.counter) return;
        
        const visibleThemes = domCache.sidePanel ?
            domCache.sidePanel.querySelectorAll('.theme-option:not([style*="display: none"])').length :
            Object.keys(THEME_CONFIGS).length;
        
        domCache.counter.textContent = `显示 ${visibleThemes}/${Object.keys(THEME_CONFIGS).length} 个主题`;
    }
    
    // 初始化搜索功能
    function initSearch() {
        if (!domCache.searchInput) return;
        
        // 使用防抖优化搜索性能
        const handleSearch = debounce((e) => {
            const searchTerm = e.target.value.toLowerCase().trim();
            const options = domCache.themeOptions;
            let visibleCount = 0;
            
            for (let i = 0; i < options.length; i++) {
                const option = options[i];
                const themeKey = option.dataset.theme;
                const theme = THEME_CONFIGS[themeKey];
                
                if (theme && (theme.name.toLowerCase().includes(searchTerm) || 
                              theme.desc.toLowerCase().includes(searchTerm))) {
                    option.style.display = 'flex';
                    visibleCount++;
                } else {
                    option.style.display = 'none';
                }
            }
            
            // 显示/隐藏无结果提示
            if (domCache.noResults) {
                domCache.noResults.style.display = visibleCount === 0 && searchTerm ? 'block' : 'none';
                if (visibleCount === 0 && searchTerm) {
                    domCache.noResults.textContent = `未找到包含"${searchTerm}"的主题`;
                }
            }
            
            updateCounter();
        }, CONFIG.DEBOUNCE_DELAY);
        
        domCache.searchInput.addEventListener('input', handleSearch);
        state.eventListeners.push({ element: domCache.searchInput, type: 'input', handler: handleSearch });
    }
    
    // 初始化事件监听 - 使用事件委托优化
    function initEventListeners() {
        // 主题选择 - 事件委托
        domCache.themeList?.addEventListener('click', (e) => {
            const themeOption = e.target.closest('.theme-option');
            if (themeOption) {
                const theme = themeOption.dataset.theme;
                setQuoteTheme(theme);
            }
        });
        
        // 随机主题
        domCache.randomBtn?.addEventListener('click', () => {
            const visibleOptions = Array.from(
                domCache.sidePanel.querySelectorAll('.theme-option:not([style*="display: none"])')
            );
            
            if (visibleOptions.length > 0) {
                const randomIndex = Math.floor(Math.random() * visibleOptions.length);
                const randomTheme = visibleOptions[randomIndex].dataset.theme;
                setQuoteTheme(randomTheme);
                
                // 添加视觉反馈
                domCache.randomBtn.style.transform = 'scale(0.95)';
                setTimeout(() => {
                    domCache.randomBtn.style.transform = 'scale(1)';
                }, 150);
            }
        });
        
        // 折叠/展开
        domCache.collapseBtn?.addEventListener('click', (e) => {
            e.stopPropagation();
            state.isCollapsed = !state.isCollapsed;
            storage.set(CONFIG.STORAGE_KEYS.SIDEBAR_COLLAPSED, state.isCollapsed);
            updateCollapseUI();
        });
        
        // 关闭
        domCache.closeBtn?.addEventListener('click', () => {
            domCache.sidePanel.style.display = 'none';
            storage.set(CONFIG.STORAGE_KEYS.SIDEBAR_VISIBLE, 'false');
        });
        
        // 全局快捷键
        const handleKeydown = (e) => {
            // 切换侧边栏
            if (e.ctrlKey && e.shiftKey && e.key === CONFIG.HOTKEYS.TOGGLE_SIDEBAR.key) {
                e.preventDefault();
                toggleSidebar();
            }
            
            // 聚焦搜索框
            if (e.ctrlKey && e.key === CONFIG.HOTKEYS.FOCUS_SEARCH.key && 
                domCache.sidePanel.style.display !== 'none') {
                e.preventDefault();
                domCache.searchInput?.focus();
            }
        };
        
        document.addEventListener('keydown', handleKeydown);
        state.eventListeners.push({ element: document, type: 'keydown', handler: handleKeydown });
    }
    
    // 更新折叠状态UI
    function updateCollapseUI() {
        if (!domCache.sidePanel) return;
        
        requestAnimationFrame(() => {
            domCache.sidePanel.classList.toggle('collapsed', state.isCollapsed);
            
            const collapseSvg = domCache.collapseBtn?.querySelector('svg');
            if (collapseSvg) {
                collapseSvg.style.transform = state.isCollapsed ? 'rotate(180deg)' : 'rotate(0deg)';
            }
            
            domCache.sidePanel.title = state.isCollapsed ? '点击展开引述块主题设置' : '';
        });
    }
    
    // 切换侧边栏显示
    function toggleSidebar() {
        if (!domCache.sidePanel) return;
        
        const isVisible = domCache.sidePanel.style.display !== 'none';
        domCache.sidePanel.style.display = isVisible ? 'none' : 'block';
        storage.set(CONFIG.STORAGE_KEYS.SIDEBAR_VISIBLE, !isVisible);
    }
    
    // 初始化拖拽功能
    function initDraggable() {
        if (!domCache.sidePanel) return;
        
        const header = domCache.sidePanel.querySelector('.sidebar-header');
        if (!header) return;
        
        header.style.cursor = 'move';
        
        const handleMouseDown = (e) => {
            if (e.target.closest('.action-btn') || e.target.classList.contains('search-input')) {
                return;
            }
            
            state.isDragging = true;
            state.dragState = {
                startX: e.clientX,
                startY: e.clientY,
                startLeft: domCache.sidePanel.offsetLeft,
                startTop: domCache.sidePanel.offsetTop
            };
            
            e.preventDefault();
        };
        
        const handleMouseMove = throttle((e) => {
            if (!state.isDragging || !state.dragState) return;
            
            const deltaX = e.clientX - state.dragState.startX;
            const deltaY = e.clientY - state.dragState.startY;
            
            domCache.sidePanel.style.left = (state.dragState.startLeft + deltaX) + 'px';
            domCache.sidePanel.style.top = (state.dragState.startTop + deltaY) + 'px';
            domCache.sidePanel.style.right = 'auto';
        }, 16); // 约60fps
        
        const handleMouseUp = () => {
            if (state.isDragging && state.dragState) {
                storage.setJSON(CONFIG.STORAGE_KEYS.SIDEBAR_POSITION, {
                    left: domCache.sidePanel.style.left,
                    top: domCache.sidePanel.style.top
                });
            }
            
            state.isDragging = false;
            state.dragState = null;
        };
        
        header.addEventListener('mousedown', handleMouseDown);
        document.addEventListener('mousemove', handleMouseMove);
        document.addEventListener('mouseup', handleMouseUp);
        
        state.eventListeners.push(
            { element: header, type: 'mousedown', handler: handleMouseDown },
            { element: document, type: 'mousemove', handler: handleMouseMove },
            { element: document, type: 'mouseup', handler: handleMouseUp }
        );
        
        // 加载保存的位置
        const savedPos = storage.getJSON(CONFIG.STORAGE_KEYS.SIDEBAR_POSITION);
        if (savedPos?.left && savedPos?.top) {
            domCache.sidePanel.style.left = savedPos.left;
            domCache.sidePanel.style.top = savedPos.top;
            domCache.sidePanel.style.right = 'auto';
        }
    }
    
    // 监听思源主题变化
    function watchSiyuanTheme() {
        if (state.themeObserver) return;
        
        state.themeObserver = new MutationObserver(() => {
            // 延迟执行，确保思源主题已完全切换
            setTimeout(() => {
                setQuoteTheme(state.currentTheme);
            }, 100);
        });
        
        state.themeObserver.observe(document.documentElement, {
            attributes: true,
            attributeFilter: ['data-theme-mode']
        });
    }
    
    // 初始化 - 核心修复：确保默认主题生效
    function initialize() {
        // 创建侧边栏
        createSidebar();
        
        // 加载状态 - 修复：先加载保存的主题，然后确保应用
        const savedTheme = storage.get(CONFIG.STORAGE_KEYS.THEME);
        state.currentTheme = savedTheme && THEME_CONFIGS[savedTheme] ? savedTheme : CONFIG.DEFAULT_THEME;
        state.isCollapsed = storage.get(CONFIG.STORAGE_KEYS.SIDEBAR_COLLAPSED) === 'true';
        
        // 关键修复：立即应用主题，不需要等待UI更新
        setQuoteTheme(state.currentTheme);
        
        // 设置侧边栏可见性
        const isVisible = storage.get(CONFIG.STORAGE_KEYS.SIDEBAR_VISIBLE) !== 'false';
        if (domCache.sidePanel) {
            domCache.sidePanel.style.display = isVisible ? 'block' : 'none';
        }
        
        // 更新UI状态
        updateCollapseUI();
        
        // 初始化功能
        initSearch();
        initEventListeners();
        initDraggable();
        watchSiyuanTheme();
        
        console.log(`引述块主题侧边栏已初始化，当前主题: ${state.currentTheme} (${THEME_CONFIGS[state.currentTheme]?.name})`);
    }
    
    // 清理函数
    function cleanup() {
        // 移除事件监听器
        state.eventListeners.forEach(({ element, type, handler }) => {
            element?.removeEventListener(type, handler);
        });
        state.eventListeners = [];
        
        // 停止观察
        if (state.themeObserver) {
            state.themeObserver.disconnect();
            state.themeObserver = null;
        }
        
        // 移除DOM元素
        if (domCache.sidePanel && domCache.sidePanel.parentNode) {
            domCache.sidePanel.parentNode.removeChild(domCache.sidePanel);
        }
        
        // 清空缓存
        domCache = {};
        state = {
            isCollapsed: false,
            currentTheme: CONFIG.DEFAULT_THEME,
            isDragging: false,
            dragState: null,
            themeObserver: null,
            eventListeners: []
        };
    }
    
    // 页面加载完成后初始化
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initialize);
    } else {
        // 立即初始化，确保主题尽快生效
        setTimeout(initialize, 50);
    }
    
    // 导出API
    window.QuoteThemeSidebar = {
        show: () => {
            if (domCache.sidePanel) {
                domCache.sidePanel.style.display = 'block';
                storage.set(CONFIG.STORAGE_KEYS.SIDEBAR_VISIBLE, 'true');
            }
        },
        hide: () => {
            if (domCache.sidePanel) {
                domCache.sidePanel.style.display = 'none';
                storage.set(CONFIG.STORAGE_KEYS.SIDEBAR_VISIBLE, 'false');
            }
        },
        toggle: toggleSidebar,
        setTheme: setQuoteTheme,
        getCurrentTheme: () => state.currentTheme,
        getAllThemes: () => ({ ...THEME_CONFIGS }),
        getThemeCount: () => Object.keys(THEME_CONFIGS).length,
        cleanup: cleanup,
        reinitialize: () => {
            cleanup();
            initialize();
        }
    };
})();