/******************************************************************
 * 思源笔记沉浸式阅读模式 - 手机端专用版
 * 功能：
 * 1. 只在移动端生效
 * 2. 只读模式自动隐藏工具栏,内容铺满屏幕
 * 3. 双击显示工具栏
 * 4. 向下滑动隐藏工具栏
 * 5. 平滑动画过渡
 * 6. ✨ 只影响主编辑器 #editor,不影响侧边栏 #sidebar 和 #model
 ******************************************************************/
(() => {
  'use strict';

  /* ============ 1. 移动端检测 ============ */
  const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent) 
    || window.innerWidth < 768;
  
  if (!isMobile) {
    console.log('[沉浸阅读] 仅在移动端生效');
    return;
  }

  /* ============ 2. 配置项 ============ */
  const CONFIG = {
    doubleTapDelay: 300,
    swipeThreshold: 50,
    animationDuration: 300,
    autoHideDelay: 0,
    debug: true
  };

  /* ============ 3. 状态管理 ============ */
  const state = {
    isReadOnly: false,
    isToolbarVisible: true,
    lastTapTime: 0,
    tapCount: 0,
    touchStartY: 0,
    touchStartX: 0,
    isSwiping: false,
    autoHideTimer: null
  };

  /* ============ 4. 工具函数 ============ */
  const log = (...args) => CONFIG.debug && console.log('[沉浸阅读]', ...args);

  // 获取需要隐藏的元素
  const getHideableElements = () => {
    return {
      toolbar: document.querySelector('.toolbar'),
      breadcrumb: document.querySelector('.protyle-breadcrumb'),
      breadcrumbBar: document.querySelector('.protyle-breadcrumb__bar'),
      top: document.querySelector('.protyle-top'),
      title: document.querySelector('.protyle-title')
    };
  };

  // 获取主编辑器（手机端是 #editor）
  const getMainEditor = () => {
    return document.getElementById('editor');
  };

  // 查找只读按钮
  const findReadOnlyButton = () => {
    const selectors = [
      '.block__icon[data-type="readonly"]',
      '[data-type="readonly"]',
      '.toolbar__item[data-type="readonly"]'
    ];
    
    for (const selector of selectors) {
      const btn = document.querySelector(selector);
      if (btn) return btn;
    }
    return null;
  };

  /* ============ 5. 样式注入 ============ */
  const injectStyles = () => {
    if (document.getElementById('immersive-reading-styles')) return;
    
    const style = document.createElement('style');
    style.id = 'immersive-reading-styles';
    style.textContent = `
      /* ========== 关键：只针对 #editor 主编辑器 ========== */
      /* 不影响 #sidebar 和 #model 侧边栏 */
      
      #editor.immersive-mode-active {
        padding-top: 0 !important;
        margin-top: 0 !important;
      }

      #editor.immersive-mode-active .protyle-content {
        padding-top: 0 !important;
        margin-top: 0 !important;
      }

      #editor.immersive-mode-active .protyle-wysiwyg {
        padding-top: 4px !important;
        margin-top: 0 !important;
      }

      #editor.immersive-mode-active .protyle-background {
        top: 0 !important;
        margin-top: 0 !important;
      }

      #editor.immersive-mode-active .protyle-scroll {
        padding-top: 0 !important;
        margin-top: 0 !important;
      }

      #editor.immersive-mode-active .protyle-wysiwyg--attr {
        margin-top: 0 !important;
        padding-top: 0 !important;
      }

      #editor.immersive-mode-active .protyle-wysiwyg > div:first-child,
      #editor.immersive-mode-active .protyle-wysiwyg > [data-node-id]:first-child {
        margin-top: 0 !important;
        padding-top: 0 !important;
      }

      #editor.immersive-mode-active::before,
      #editor.immersive-mode-active .protyle-content::before,
      #editor.immersive-mode-active .protyle-wysiwyg::before {
        display: none !important;
      }

      /* ========== 工具栏隐藏/显示样式 ========== */
      /* 只针对顶部主工具栏,不影响侧边栏内的 toolbar */
      body.immersive-reading-mode > .toolbar,
      body.immersive-reading-mode #editor .protyle-breadcrumb,
      body.immersive-reading-mode #editor .protyle-breadcrumb__bar,
      body.immersive-reading-mode #editor .protyle-top,
      body.immersive-reading-mode #editor .protyle-title {
        position: fixed !important;
        top: 0 !important;
        left: 0 !important;
        right: 0 !important;
        z-index: 999 !important;
        background: var(--b3-theme-background) !important;
        transition: transform ${CONFIG.animationDuration}ms ease-out, 
                    opacity ${CONFIG.animationDuration}ms ease-out !important;
      }

      /* 工具栏隐藏状态 */
      body.immersive-reading-mode > .toolbar.immersive-hidden,
      body.immersive-reading-mode #editor .protyle-breadcrumb.immersive-hidden,
      body.immersive-reading-mode #editor .protyle-breadcrumb__bar.immersive-hidden,
      body.immersive-reading-mode #editor .protyle-top.immersive-hidden,
      body.immersive-reading-mode #editor .protyle-title.immersive-hidden {
        transform: translateY(-100%) !important;
        opacity: 0 !important;
        pointer-events: none !important;
      }

      /* 工具栏显示状态 */
      body.immersive-reading-mode > .toolbar.immersive-visible,
      body.immersive-reading-mode #editor .protyle-breadcrumb.immersive-visible,
      body.immersive-reading-mode #editor .protyle-breadcrumb__bar.immersive-visible,
      body.immersive-reading-mode #editor .protyle-top.immersive-visible,
      body.immersive-reading-mode #editor .protyle-title.immersive-visible {
        transform: translateY(0) !important;
        opacity: 1 !important;
        pointer-events: auto !important;
      }

      /* ========== 确保侧边栏不受影响 ========== */
      /* 明确保护侧边栏的工具栏 */
      #sidebar .toolbar,
      #model .toolbar {
        position: relative !important;
        transform: none !important;
        opacity: 1 !important;
      }

      #sidebar,
      #model,
      #sidebar .protyle,
      #model .protyle {
        padding-top: initial !important;
        margin-top: initial !important;
      }

      #sidebar .protyle-content,
      #model .protyle-content {
        padding-top: initial !important;
        margin-top: initial !important;
      }

      /* 确保滚动流畅 */
      .protyle-content {
        overflow-y: auto !important;
        -webkit-overflow-scrolling: touch !important;
      }

      /* 动画效果 */
      .toolbar,
      .protyle-breadcrumb,
      .protyle-breadcrumb__bar,
      .protyle-top,
      .protyle-title {
        transition: opacity ${CONFIG.animationDuration}ms ease-out;
      }
    `;
    document.head.appendChild(style);
    log('✅ 样式已注入（手机端专用版）');
  };

  /* ============ 6. 核心功能：隐藏工具栏 ============ */
  const hideToolbar = () => {
    if (!state.isReadOnly || !state.isToolbarVisible) return;

    const elements = getHideableElements();
    const mainEditor = getMainEditor();

    // 隐藏工具栏元素
    Object.values(elements).forEach(el => {
      if (el) {
        el.classList.remove('immersive-visible');
        el.classList.add('immersive-hidden');
      }
    });

    // 给主编辑器添加沉浸模式标记
    if (mainEditor) {
      mainEditor.classList.add('immersive-mode-active');
      log('✅ 主编辑器 #editor 已添加沉浸模式');
    } else {
      log('⚠️ 未找到主编辑器 #editor');
    }

    // 给body添加沉浸模式类
    document.body.classList.add('immersive-reading-mode');

    state.isToolbarVisible = false;
    log('🙈 工具栏已隐藏,内容铺满全屏（侧边栏不受影响）');
  };

  /* ============ 7. 核心功能：显示工具栏 ============ */
  const showToolbar = () => {
    if (!state.isReadOnly || state.isToolbarVisible) return;

    const elements = getHideableElements();
    const mainEditor = getMainEditor();

    // 显示工具栏元素
    Object.values(elements).forEach(el => {
      if (el) {
        el.classList.remove('immersive-hidden');
        el.classList.add('immersive-visible');
      }
    });

    // 移除主编辑器的沉浸模式标记
    if (mainEditor) {
      mainEditor.classList.remove('immersive-mode-active');
      log('✅ 主编辑器 #editor 已移除沉浸模式');
    }

    // 移除body的沉浸模式类
    document.body.classList.remove('immersive-reading-mode');

    state.isToolbarVisible = true;
    log('👀 工具栏已显示');

    // 自动隐藏（如果配置了）
    if (CONFIG.autoHideDelay > 0) {
      clearTimeout(state.autoHideTimer);
      state.autoHideTimer = setTimeout(() => {
        log('⏰ 自动隐藏触发');
        hideToolbar();
      }, CONFIG.autoHideDelay);
    }
  };

  /* ============ 8. 双击检测 ============ */
  const handleTap = (e) => {
    if (!state.isReadOnly) return;

    const now = Date.now();
    const timeSinceLastTap = now - state.lastTapTime;

    if (timeSinceLastTap > CONFIG.doubleTapDelay) {
      state.tapCount = 0;
    }

    state.tapCount++;
    state.lastTapTime = now;

    if (state.tapCount === 2 && timeSinceLastTap < CONFIG.doubleTapDelay) {
      e.preventDefault();
      state.tapCount = 0;
      
      // 切换显示/隐藏
      if (state.isToolbarVisible) {
        hideToolbar();
      } else {
        showToolbar();
      }
      
      log(`🎯 双击检测成功,${state.isToolbarVisible ? '显示' : '隐藏'}工具栏`);
    }

    if (state.tapCount > 2) {
      state.tapCount = 0;
    }
  };

  /* ============ 9. 滑动检测 ============ */
  const handleTouchStart = (e) => {
    if (!state.isReadOnly) return;

    state.touchStartY = e.touches[0].clientY;
    state.touchStartX = e.touches[0].clientX;
    state.isSwiping = false;
  };

  const handleTouchMove = (e) => {
    if (!state.isReadOnly || state.isSwiping) return;

    const touchY = e.touches[0].clientY;
    const touchX = e.touches[0].clientX;
    const deltaY = touchY - state.touchStartY;
    const deltaX = Math.abs(touchX - state.touchStartX);

    // 确保是垂直滑动
    if (Math.abs(deltaY) > deltaX && Math.abs(deltaY) > 10) {
      state.isSwiping = true;
      
      // 向下滑动且工具栏可见 -> 隐藏
      if (deltaY > CONFIG.swipeThreshold && state.isToolbarVisible) {
        log('👇 检测到向下滑动,隐藏工具栏');
        hideToolbar();
      }
      // 向上滑动且工具栏隐藏 -> 显示
      else if (deltaY < -CONFIG.swipeThreshold && !state.isToolbarVisible) {
        log('👆 检测到向上滑动,显示工具栏');
        showToolbar();
      }
    }
  };

  const handleTouchEnd = () => {
    state.isSwiping = false;
  };

  /* ============ 10. 事件监听管理 ============ */
  let eventListenersAttached = false;

  const attachEventListeners = () => {
    if (eventListenersAttached) return;

    document.body.addEventListener('touchstart', handleTap, { passive: false });
    document.body.addEventListener('touchstart', handleTouchStart, { passive: true });
    document.body.addEventListener('touchmove', handleTouchMove, { passive: true });
    document.body.addEventListener('touchend', handleTouchEnd, { passive: true });

    eventListenersAttached = true;
    log('✅ 事件监听已添加');
  };

  const detachEventListeners = () => {
    if (!eventListenersAttached) return;

    document.body.removeEventListener('touchstart', handleTap);
    document.body.removeEventListener('touchstart', handleTouchStart);
    document.body.removeEventListener('touchmove', handleTouchMove);
    document.body.removeEventListener('touchend', handleTouchEnd);

    eventListenersAttached = false;
    log('❌ 事件监听已移除');
  };

  /* ============ 11. 模式切换 ============ */
  const enterReadOnlyMode = () => {
    log('🔒 进入只读模式');
    state.isReadOnly = true;
    state.isToolbarVisible = true;
    attachEventListeners();
    
    // 延迟隐藏
    setTimeout(() => {
      hideToolbar();
    }, 500);
  };

  const exitReadOnlyMode = () => {
    log('✏️ 退出只读模式');
    state.isReadOnly = false;
    detachEventListeners();
    clearTimeout(state.autoHideTimer);
    
    const elements = getHideableElements();
    const mainEditor = getMainEditor();

    // 恢复工具栏显示
    Object.values(elements).forEach(el => {
      if (el) {
        el.classList.remove('immersive-hidden', 'immersive-visible');
      }
    });

    // 移除主编辑器的沉浸模式标记
    if (mainEditor) {
      mainEditor.classList.remove('immersive-mode-active');
    }

    // 移除body的沉浸模式类
    document.body.classList.remove('immersive-reading-mode');

    state.isToolbarVisible = true;
  };

  /* ============ 12. 状态监听 ============ */
  let buttonObserver = null;

  const checkReadOnlyStatus = () => {
    const btn = findReadOnlyButton();
    
    if (!btn) {
      if (state.isReadOnly) {
        exitReadOnlyMode();
      }
      return;
    }

    // 重新观察按钮
    if (buttonObserver) {
      buttonObserver.disconnect();
    }
    buttonObserver = new MutationObserver(() => {
      log('🔄 按钮状态变化');
      checkReadOnlyStatus();
    });
    buttonObserver.observe(btn, {
      attributes: true,
      attributeFilter: ['data-subtype', 'class', 'aria-label']
    });

    // 判断是否只读模式
    const isNowReadOnly = btn.dataset.subtype === 'lock' 
      || btn.classList.contains('toolbar__item--active')
      || btn.classList.contains('block__icon--active');

    if (isNowReadOnly && !state.isReadOnly) {
      enterReadOnlyMode();
    } else if (!isNowReadOnly && state.isReadOnly) {
      exitReadOnlyMode();
    }
  };

  /* ============ 13. 页面变化监听 ============ */
  let pageObserver = null;

  const observePageChanges = () => {
    if (pageObserver) {
      pageObserver.disconnect();
    }

    pageObserver = new MutationObserver((mutations) => {
      const shouldCheck = mutations.some(mutation =>
        Array.from(mutation.addedNodes).some(node =>
          node.nodeType === 1 && (
            node.classList?.contains('protyle') ||
            node.querySelector?.('.protyle') ||
            node.classList?.contains('toolbar') ||
            node.querySelector?.('[data-type="readonly"]')
          )
        )
      );

      if (shouldCheck) {
        log('🔄 页面结构变化,重新检查');
        setTimeout(checkReadOnlyStatus, 100);
      }
    });

    pageObserver.observe(document.body, {
      childList: true,
      subtree: true
    });
  };

  /* ============ 14. 初始化 ============ */
  const init = () => {
    log('🚀 沉浸式阅读模式初始化（手机端专用版）');
    log(`📱 设备: ${navigator.userAgent.match(/Android|iPhone|iPad/)?.[0] || 'Unknown'}`);
    log(`📐 屏幕: ${window.innerWidth}x${window.innerHeight}`);
    
    injectStyles();
    checkReadOnlyStatus();
    observePageChanges();
    
    // 检查关键元素
    const mainEditor = getMainEditor();
    const sidebar = document.getElementById('sidebar');
    const model = document.getElementById('model');
    
    log('🔍 DOM 结构检查:');
    log(`  - 主编辑器 #editor: ${mainEditor ? '✅' : '❌'}`);
    log(`  - 侧边栏 #sidebar: ${sidebar ? '✅' : '❌'}`);
    log(`  - 侧边栏 #model: ${model ? '✅' : '❌'}`);
    
    log('✅ 初始化完成');
    log('💡 使用方法:');
    log('   1️⃣ 点击锁图标进入只读模式');
    log('   2️⃣ 工具栏自动隐藏,内容铺满屏幕');
    log('   3️⃣ 双击屏幕切换显示/隐藏');
    log('   4️⃣ 向下滑动隐藏,向上滑动显示');
    log('   ✨ #sidebar 和 #model 侧边栏不受影响！');
  };

  // 等待DOM加载
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // 页面可见性变化
  document.addEventListener('visibilitychange', () => {
    if (!document.hidden) {
      log('👀 页面重新可见');
      checkReadOnlyStatus();
    }
  });

  /* ============ 15. 调试接口 ============ */
  window.__immersiveReading = {
    getState: () => ({ ...state }),
    show: showToolbar,
    hide: hideToolbar,
    toggle: () => state.isToolbarVisible ? hideToolbar() : showToolbar(),
    refresh: checkReadOnlyStatus,
    setDebug: (val) => { CONFIG.debug = val; },
    setAutoHide: (ms) => { CONFIG.autoHideDelay = ms; },
    
    // 检查DOM结构（手机端版本）
    checkDOM: () => {
      console.log('=== 思源笔记 DOM 结构检查（手机端）===\n');
      
      const mainEditor = document.getElementById('editor');
      const sidebar = document.getElementById('sidebar');
      const model = document.getElementById('model');
      
      console.log('主编辑器 #editor:', mainEditor ? '✅ 找到' : '❌ 未找到');
      if (mainEditor) {
        console.log('  - 沉浸模式:', mainEditor.classList.contains('immersive-mode-active') ? '已激活 ✨' : '未激活');
        console.log('  - 类名:', mainEditor.className);
      }
      
      console.log('\n侧边栏 #sidebar:', sidebar ? '✅ 找到' : '❌ 未找到');
      if (sidebar) {
        console.log('  - 类名:', sidebar.className);
      }
      
      console.log('\n侧边栏 #model:', model ? '✅ 找到' : '❌ 未找到');
      if (model) {
        console.log('  - 类名:', model.className);
      }
      
      console.log('\n当前模式:', state.isReadOnly ? '只读 🔒' : '编辑 ✏️');
      console.log('工具栏状态:', state.isToolbarVisible ? '显示 👀' : '隐藏 🙈');
      console.log('body 类名:', document.body.className);
    }
  };

  log('🛠️ 调试接口: window.__immersiveReading');
})();