# 跳过异常继续执行

# 跳过异常继续执行

　　在数据处理过程中，常常会遇到因文件缺失、格式错误或权限问题导致的异常情况。为了确保程序的健壮性和连续性，可以采用“跳过异常继续执行”的策略，即在捕获到异常时忽略该次操作并继续后续流程。

## 示例代码（Python）

　　以下是一个使用 `pandas` 读取多个日期股票数据文件的示例，若某一天的数据文件不存在或无法读取，则跳过该文件，继续处理其他文件：

```python
import pandas as pd

# 定义日期范围（例如：2016年10月10日至2016年11月13日）
dates = range(20161010, 20161114)

# 存储所有成功读取的数据片段
pieces = []

# 遍历每个日期
for date in dates:
    try:
        # 尝试读取对应日期的CSV文件（编码为GBK，适用于中文环境）
        data = pd.read_csv(
            f'A_stock/overview-push-{date}/stockoverview.csv',
            encoding='gbk'
        )
        pieces.append(data)
        print(f"✅ 成功读取日期 {date} 的数据")
    
    except OSError:
        # 若文件不存在或无法访问，跳过该文件并输出提示
        print(f"⚠️  跳过日期 {date} 的数据（文件不存在或访问失败）")
        continue
    
    except Exception as e:
        # 捕获其他未知异常（如文件格式错误等），同样跳过
        print(f"❌ 日期 {date} 出现未知错误: {e}")
        continue

# 合并所有成功读取的数据片段
if pieces:
    combined_data = pd.concat(pieces, ignore_index=True)
    print(f"📊 最终合并数据形状: {combined_data.shape}")
else:
    print("⚠️ 未读取到任何有效数据")

```

## 关键要点说明

- **异常类型处理**：
  - `OSError`：常见于文件不存在、路径错误或权限不足。
  - `Exception`：用于捕获其他未预期的错误（如编码不匹配、空文件等）。
- **日志记录**：通过 `print()` 输出每一步的状态，便于调试和监控。
- **结果验证**：合并前检查是否有数据片段，避免空DataFrame引发后续错误。
- **可扩展性**：此模式适用于批量读取多个文件、API请求、数据库查询等场景。

> ✅ 推荐在生产环境中结合日志系统（如 `logging` 模块）进一步增强可观测性。
