# 捕获异常

# 捕获异常

　　在编程中，异常处理是一种重要的机制，用于应对程序运行时可能出现的错误或意外情况。通过捕获和处理异常，可以避免程序因未预期的问题而崩溃，同时提升用户体验和代码的健壮性。

## 基本语法结构

　　Python 中使用 `try...except...else...finally` 语句来实现异常处理：

```python
try:
    # 可能引发异常的代码块
    print("执行正常操作")
    raise NetworkError("网络连接失败")  # 示例：主动抛出异常
except Exception as e:
    # 处理异常的代码块
    print(f"捕获到异常：{e}")
    print("异常处理逻辑")
else:
    # 当没有异常发生时执行的代码块（可选）
    print("无异常，执行 else 分支")
finally:
    # 无论是否有异常都会执行的代码块（可选）
    print("清理资源或收尾工作")
```

### 关键说明：

- **​`try`​**：包含可能产生异常的代码。
- **​`except`​**：指定要捕获的异常类型及其处理逻辑。若不指定具体异常类型，则会捕获所有异常（如 `Exception`）。
- **​`else`​**：仅在 `try` 块中没有异常时执行。
- **​`finally`​**：无论是否发生异常，都会执行，常用于释放资源（如关闭文件、数据库连接等）。

## 实际应用场景示例

### 场景1：文件读取异常处理

```python
try:
    with open("data.txt", "r", encoding="utf-8") as file:
        content = file.read()
        print(content)
except FileNotFoundError:
    print("文件不存在，请检查路径")
except PermissionError:
    print("没有权限访问该文件")
except Exception as e:
    print(f"其他错误：{e}")
```

### 场景2：网络请求异常处理

```python
import requests

try:
    response = requests.get("https://api.example.com/data", timeout=5)
    response.raise_for_status()  # 如果状态码不是 200，则抛出异常
    print("请求成功")
except requests.exceptions.ConnectionError:
    print("网络连接失败")
except requests.exceptions.Timeout:
    print("请求超时")
except requests.exceptions.HTTPError as e:
    print(f"HTTP 错误：{e}")
except Exception as e:
    print(f"未知错误：{e}")
```

## 最佳实践建议

1. **明确异常类型**：尽量捕获具体的异常类型，而不是泛型异常（如 `Exception`），以便更精准地处理问题。
2. **避免空的 **​**​`except`​**​ ** 块**：确保每次异常都有对应的处理逻辑，避免忽略错误。
3. **合理使用 **​**​`finally`​**：用于确保资源释放，例如关闭文件、数据库连接等。
4. **记录日志**：在生产环境中，应将异常信息记录到日志文件中，便于排查问题。
5. **用户友好提示**：向用户提供清晰、易懂的错误信息，而非技术性的堆栈跟踪。

　　通过合理的异常处理机制，可以显著提高程序的稳定性和可维护性。
