# 异常重试-重头执行

# 异常重试 - 重头执行（优化版）

　　在软件开发中，某些操作可能因临时性问题（如网络波动、服务短暂不可用、资源竞争等）而失败。为了提升程序的健壮性和用户体验，我们通常需要实现**异常重试机制**，即当某个函数或操作失败时自动重新执行，直到成功或达到预设条件为止。

　　本文将详细介绍如何使用 Python 的 `retrying` 模块来实现灵活且可控的异常重试策略，涵盖常见场景和最佳实践。

---

## 一、安装与基础用法

### 安装

```bash
pip install retrying
```

> ✅ 支持 Python 3.4+ 版本。

### 基础装饰器：`@retry`

　　最简单的用法是直接使用 `@retry` 装饰器，它会无限制地重试失败的操作：

```python
from retrying import retry
import random

@retry
def do_something_unreliable():
    if random.randint(0, 10) > 1:
        raise IOError("Brokensauce, everything is hose!!!")
    else:
        return "Awesomesauce!"
```

　　⚠️ 注意：这种方式不会等待，会持续快速重试，可能导致系统压力过大，建议配合停止条件使用。

---

## 二、常用配置参数详解

### 1. 设置最大重试次数：`stop_max_attempt_number`

```python
@retry(stop_max_attempt_number=7)
def stop_after_7_attempts():
    print("Stopping after 7 attempts")
    raise Exception("Simulated failure")
```

- 最多尝试 7 次后终止。
- 可用于防止无限循环导致的服务雪崩。

### 2. 设置总超时时间：`stop_max_delay`

　　单位为毫秒（ms），超过该时间则不再重试：

```python
@retry(stop_max_delay=10000)  # 10秒内最多重试
def stop_after_10_seconds():
    print("Stopping after 10 seconds")
    raise Exception("Simulated failure")
```

### 3. 固定间隔等待：`wait_fixed`

　　每次重试之间固定等待指定时间（毫秒）：

```python
@retry(wait_fixed=2000)
def wait_2_seconds():
    print("Wait 2 seconds between retries")
    raise Exception("Simulated failure")
```

### 4. 随机间隔等待：`wait_random_min`, `wait_random_max`

　　模拟更真实的网络延迟行为：

```python
@retry(wait_random_min=1000, wait_random_max=5000)
def wait_random_1_to_5_seconds():
    print("Randomly wait 1 to 5 seconds between retries")
    raise Exception("Simulated failure")
```

### 5. 指数退避策略：`wait_exponential_multiplier`, `wait_exponential_max`

　　适用于分布式服务调用、API 请求等场景，避免集中式重试风暴：

```python
@retry(
    wait_exponential_multiplier=1000,   # 初始等待时间为 1 秒
    wait_exponential_max=10000          # 最大等待时间为 10 秒
)
def exponential_backoff():
    print("Wait 2^x * 1000 ms between retries, up to 10s")
    raise Exception("Simulated failure")
```

　　输出示例：

```
Wait 2^x * 1000 ms between retries, up to 10s
1504110314
Wait 2^x * 1000 ms between retries, up to 10s
1504110316
...
```

---

## 三、按异常类型决定是否重试：`retry_on_exception`

　　通过自定义函数判断是否应触发重试，例如只对 IO 错误进行重试：

```python
def retry_if_io_error(exception):
    """仅当发生 IOError 时重试"""
    return isinstance(exception, IOError)

@retry(retry_on_exception=retry_if_io_error)
def might_io_error():
    print("Retry forever with no wait if an IOError occurs, raise any other errors")
    raise IOError("Network issue")  # 此类异常会被重试
```

　　如果希望其他异常也被包装成 `RetryError` 抛出（便于统一处理），可启用 `wrap_exception=True`：

```python
@retry(
    retry_on_exception=retry_if_io_error,
    wrap_exception=True
)
def only_raise_retry_error_when_not_io_error():
    print("Retry forever with no wait if an IOError occurs, raise any other errors wrapped in RetryError")
    raise ValueError("Unexpected error")  # 将被包装为 RetryError
```

---

## 四、根据返回结果决定是否重试：`retry_on_result`

　　有时业务逻辑中的空值（None）也可能表示“暂时不可用”，此时可根据返回值决定是否重试：

```python
def retry_if_result_none(result):
    """若返回值为 None，则重试"""
    return result is None

@retry(retry_on_result=retry_if_result_none)
def get_result():
    print("Retry forever ignoring exceptions with no wait if return value is None")
    return None  # 会导致重试
```

　　此模式特别适合调用远程接口获取数据但可能暂时未生成的情况（如缓存未命中、异步任务未完成等）。

---

## 五、组合策略推荐（实战案例）

　　实际项目中往往需要多种策略组合，例如：

```python
@retry(
    stop_max_attempt_number=5,
    wait_exponential_multiplier=1000,
    wait_exponential_max=10000,
    retry_on_exception=lambda e: isinstance(e, (IOError, ConnectionError)),
    wrap_exception=True
)
def call_remote_api(url):
    response = requests.get(url)
    if response.status_code != 200:
        raise ConnectionError(f"HTTP {response.status_code}")
    return response.json()
```

　　✅ 说明：

- 最多尝试 5 次；
- 使用指数退避（首次 1s，后续逐步增长至 10s）；
- 对 IO 错误和连接错误自动重试；
- 其他异常将被封装为 `RetryError`，便于捕获和日志记录。

---

## 六、总结建议

|场景|推荐策略|
| -------------------| ------------------|
|短暂网络抖动|`wait_random_min/max` + `stop_max_attempt_number`|
|API 调用失败|`wait_exponential_*` + `retry_on_exception`|
|缓存/数据库查询为空|`retry_on_result`|
|高并发下防雪崩|结合 `stop_max_delay` 和 `wait_fixed` 控制频率|

　　📌 **关键点提醒**：

- 不要盲目无限重试，必须设置合理的上限；
- 合理利用等待策略，减少对下游系统的冲击；
- 明确区分哪些异常应该重试，哪些不应该；
- 在生产环境中建议结合日志监控，及时发现并定位频繁失败的问题。

---

　　📌 如需进一步定制行为（如回调钩子、重试次数统计等），可参考官方文档扩展使用方式。
