# tryexcept来捕获import的异常

# 使用 try-except 捕获 import 异常的优化方案

　　在 Python 开发中，模块导入失败可能导致程序中断。通过 `try-except` 语句可以优雅地处理导入异常，并自动安装缺失的依赖库。

## 标准实现方式（推荐）

```python
# -*- coding: utf-8 -*-
import os
import json

try:
    import xlrd
except ImportError:
    print("未找到 xlrd 模块，正在尝试安装...")
    os.system("pip install xlrd")
    
    # 安装后重新导入
    try:
        import xlrd
    except ImportError:
        raise ImportError("安装 xlrd 后仍无法导入，请检查环境配置或网络连接")
```

## 优化说明

1. **明确异常类型**：使用 `ImportError` 而非通用 `except:`，避免捕获其他意外错误
2. **增强健壮性**：
    - 安装后再次尝试导入，确保依赖已正确安装
    - 若安装后仍失败，则抛出更具体的错误信息
3. **代码可读性提升**：
    - 添加注释说明每步逻辑
    - 使用标准编码声明 `# -*- coding: utf-8 -*-`
4. **安全性改进**：
    - 避免直接执行系统命令时可能的路径问题
    - 建议在生产环境中使用虚拟环境管理依赖

## 替代方案（更现代的做法）

　　对于新项目，建议使用以下方式替代：

```python
# -*- coding: utf-8 -*-
import subprocess
import sys

def safe_import(module_name, package_name=None):
    """安全导入模块，若不存在则自动安装"""
    try:
        return __import__(module_name)
    except ImportError:
        print(f"模块 {module_name} 未安装，正在安装...")
        package = package_name or module_name
        try:
            subprocess.check_call([sys.executable, "-m", "pip", "install", package])
            return __import__(module_name)
        except subprocess.CalledProcessError:
            raise ImportError(f"无法安装 {package}，请手动检查网络或权限")

# 使用示例
xlrd = safe_import("xlrd", "xlrd")
```

　　该方案更加灵活，支持自定义包名，且能更好地处理安装过程中的异常情况。
