import os
import glob
import threading
import time
import fnmatch
from pathlib import Path
from chat_api import ApiTask
from concurrent.futures import ThreadPoolExecutor, as_completed

# 配置开关
CONFIG = {
    "ENABLE_BACKUP": False,  # 是否备份原文件 (True: 备份, False: 不备份)
    "MAX_WORKERS": 3,       # 最大并发线程数
    "FILE_EXTENSIONS": [".md", ".txt", ".rst"],  # 搜索的文件后缀列表
    "IGNORE_FILES": [        # 忽略的文件名列表（支持通配符）
        "README.md",
        "LICENSE*",
        "CHANGELOG*",
        "*.backup",
        "*.tmp",
        "*.log"
    ],
    "IGNORE_DIRS": [         # 忽略的目录名列表（支持通配符）
        ".git",
        ".vscode",
        ".idea",
        "__pycache__",
        "node_modules",
        "*.egg-info",
        "build",
        "dist",
        "temp",
        "tmp"
    ]
}

"""
配置说明：
1. ENABLE_BACKUP: 控制是否为每个处理的文件创建备份
   - True: 在处理前创建 .backup 备份文件
   - False: 直接覆盖原文件，不创建备份
   
2. MAX_WORKERS: 设置最大并发线程数，建议不超过5个

3. FILE_EXTENSIONS: 指定要搜索和处理的文件后缀列表
   - 默认支持: .md (Markdown), .txt (纯文本), .rst (reStructuredText)
   - 可以添加其他后缀，如: [".md", ".txt", ".doc", ".html"]
   - 注意：后缀需要包含点号，如 ".md" 而不是 "md"

4. IGNORE_FILES: 忽略的文件名列表（支持通配符）
   - 支持精确文件名："README.md"
   - 支持通配符："LICENSE*" 匹配以LICENSE开头的所有文件
   - 支持后缀匹配："*.backup" 匹配所有.backup后缀文件
   - 常用忽略：备份文件、日志文件、临时文件等

5. IGNORE_DIRS: 忽略的目录名列表（支持通配符）
   - 支持精确目录名：".git", "node_modules"
   - 支持通配符："*.egg-info" 匹配所有以.egg-info结尾的目录
   - 常用忽略：版本控制目录、IDE目录、编译目录等

使用建议：
- 首次使用或重要文件处理时，建议设置 ENABLE_BACKUP=True
- 批量处理且确认无需备份时，可设置 ENABLE_BACKUP=False 节省存储空间
- 根据实际需要调整 FILE_EXTENSIONS 列表，只处理需要的文件类型
- 合理配置 IGNORE_FILES 和 IGNORE_DIRS 列表，避免处理不必要的文件
- 通配符示例："temp*" 匹配 temp 开头的所有文件/目录，"*log" 匹配 log 结尾的所有文件/目录
"""
def should_ignore(item_name, ignore_patterns):
    """
    检查文件名或目录名是否应该被忽略
    
    Args:
        item_name (str): 文件名或目录名
        ignore_patterns (list): 忽略模式列表（支持通配符）
    
    Returns:
        bool: True 表示应该忽略，False 表示不忽略
    """
    if not ignore_patterns:
        return False
        
    for pattern in ignore_patterns:
        # 使用 fnmatch 进行通配符匹配，不区分大小写
        if fnmatch.fnmatch(item_name.lower(), pattern.lower()):
            return True
    
    return False

def find_files_by_extensions(directory, extensions=[".md"], ignore_files=None, ignore_dirs=None):
    """
    根据指定的文件后缀搜索文件，并支持忽略指定的文件和目录
    
    Args:
        directory (str): 要搜索的根目录路径
        extensions (list): 文件后缀列表，默认[".md"]
        ignore_files (list): 忽略的文件名模式列表
        ignore_dirs (list): 忽略的目录名模式列表
    
    Returns:
        list: 包含所有匹配文件路径的列表
    """
    found_files = []
    ignored_files = []
    ignored_dirs = []
    
    # 默认忽略列表（如果没有提供）
    if ignore_files is None:
        ignore_files = []
    if ignore_dirs is None:
        ignore_dirs = []
    
    # 验证后缀格式
    valid_extensions = []
    for ext in extensions:
        if not ext.startswith('.'):
            ext = '.' + ext  # 自动添加点号
        valid_extensions.append(ext.lower())
    
    print(f"搜索文件后缀: {valid_extensions}")
    if ignore_files:
        print(f"忽略文件模式: {ignore_files}")
    if ignore_dirs:
        print(f"忽略目录模式: {ignore_dirs}")
    
    # 使用os.walk递归搜索，支持目录忽略
    for root, dirs, files in os.walk(directory):
        # 检查当前目录是否应该被忽略
        current_dir_name = os.path.basename(root)
        if should_ignore(current_dir_name, ignore_dirs):
            ignored_dirs.append(root)
            dirs.clear()  # 清空 dirs 列表，不再递归进入子目录
            continue
        
        # 从 dirs中移除需要忽略的子目录（避免递归进入）
        dirs_to_remove = []
        for dir_name in dirs:
            if should_ignore(dir_name, ignore_dirs):
                dirs_to_remove.append(dir_name)
                ignored_dirs.append(os.path.join(root, dir_name))
        
        for dir_name in dirs_to_remove:
            dirs.remove(dir_name)
        
        # 处理当前目录下的文件
        for file_name in files:
            file_path = os.path.join(root, file_name)
            file_obj = Path(file_path)
            
            # 检查文件是否应该被忽略
            if should_ignore(file_name, ignore_files):
                ignored_files.append(file_path)
                continue
            
            # 检查文件后缀是否匹配
            if file_obj.suffix.lower() in valid_extensions:
                found_files.append(file_path)
    
    # 统计忽略信息
    if ignored_files or ignored_dirs:
        print(f"\n忽略统计:")
        if ignored_dirs:
            print(f"  忽略目录: {len(ignored_dirs)} 个")
            for ignored_dir in ignored_dirs:
                print(f"    - {ignored_dir}")
        if ignored_files:
            print(f"  忽略文件: {len(ignored_files)} 个")
            for ignored_file in ignored_files:
                print(f"    - {ignored_file}")
    
    # 去除重复并按路径排序
    found_files = sorted(list(set(found_files)))
    
    return found_files

def print_found_files(found_files):
    """
    打印所有找到的文件信息
    
    Args:
        found_files (list): 文件路径列表
    """
    if not found_files:
        print("未找到任何文件")
        return
    
    print(f"\n共找到 {len(found_files)} 个文件:")
    print("=" * 50)
    
    # 按文件后缀分组统计
    ext_count = {}
    for file_path in found_files:
        ext = Path(file_path).suffix.lower()
        ext_count[ext] = ext_count.get(ext, 0) + 1
    
    print(f"文件类型统计:")
    for ext, count in sorted(ext_count.items()):
        print(f"  {ext}: {count} 个文件")
    
    print("\n文件详情:")
    for i, file_path in enumerate(found_files, 1):
        # 获取文件信息
        file_obj = Path(file_path)
        file_name = file_obj.name
        file_dir = file_obj.parent
        file_ext = file_obj.suffix
        file_size = file_obj.stat().st_size if file_obj.exists() else 0
        
        print(f"{i}. 文件名: {file_name} ({file_ext})")
        print(f"   完整路径: {file_path}")
        print(f"   所在目录: {file_dir}")
        print(f"   文件大小: {file_size} 字节")
def process_single_file(file_path, thread_id, enable_backup=True):
    """
    处理单个MD文件：可选择备份原文件后生成新文件
    
    Args:
        file_path (str): 文件路径
        thread_id (int): 线程ID
        enable_backup (bool): 是否备份原文件，默认True
    
    Returns:
        dict: 处理结果
    """
    backup_path = None
    try:
        print(f"[线程{thread_id}] 开始处理文件: {file_path}")
        if enable_backup:
            print(f"[线程{thread_id}] 备份模式: 启用")
        else:
            print(f"[线程{thread_id}] 备份模式: 禁用")
        
        # 读取原始文件内容
        with open(file_path, 'r', encoding='utf-8') as f:
            original_content = f.read()
        
        print(f"[线程{thread_id}] 原始文件内容长度: {len(original_content)} 字符")
        # print(f"[线程{thread_id}] 原始内容前100字符: {original_content[:100]}...")
        
        # 根据配置决定是否创建备份文件
        if enable_backup:
            backup_path = file_path + '.backup'
            with open(backup_path, 'w', encoding='utf-8') as f:
                f.write(original_content)
            print(f"[线程{thread_id}] 已创建备份文件: {backup_path}")
        else:
            print(f"[线程{thread_id}] 跳过备份步骤")
        
        # 准备API请求数据
        api_task = ApiTask("123456789")
        data = {
            "query": f"请根据<node_data>的file_content字段的MD文件，针对文档原有内容，补全完善编写并优化MD内容，不保留原始文档链接查看，不保留 旧版本，不保留新版",
            "agent": "auto",
            "bot": "通用智能体",
            "prompt": "默认通用",
            "stream": True,
            "user_data": {},
            "node_data": {"file_content": original_content},
            "node_tag": [],
            "user_tag": [],
            "history": [],
            "file_data": [],
        }
        
        print(f"[线程{thread_id}] 正在调用AI接口...")
        rst_data = api_task.chat_ai(data)
        
        print(f"[线程{thread_id}] AI返回内容长度: {len(rst_data) if rst_data else 0} 字符")
        if rst_data:
            # print(f"[线程{thread_id}] AI返回内容前100字符: {rst_data[:100]}...")
            
            # 写入新的优化内容到原文件
            with open(file_path, 'w', encoding='utf-8') as f:
                f.write(rst_data)
            print(f"[线程{thread_id}] 新内容已写入原文件: {file_path}")
            
            # 验证新文件内容
            with open(file_path, 'r', encoding='utf-8') as f:
                new_content = f.read()
            print(f"[线程{thread_id}] 验证新文件长度: {len(new_content)} 字符")
            
            # 对比原始内容和新内容
            content_changed = original_content != new_content
            print(f"[线程{thread_id}] 内容是否发生变化: {'是' if content_changed else '否'}")
            
            # 返回结果包含备份信息
            result = {
                "success": True,
                "file_path": file_path,
                "thread_id": thread_id,
                "message": "处理成功",
                "original_length": len(original_content),
                "new_length": len(rst_data),
                "content_changed": content_changed,
                "backup_enabled": enable_backup
            }
            
            # 如果启用了备份，添加备份路径
            if enable_backup and backup_path:
                result["backup_path"] = backup_path
                
            return result
        else:
            print(f"[线程{thread_id}] AI接口返回空结果")
            # 如果AI处理失败且启用了备份，删除备份文件
            if enable_backup and backup_path and os.path.exists(backup_path):
                os.remove(backup_path)
                print(f"[线程{thread_id}] 已删除备份文件: {backup_path}")
            return {
                "success": False,
                "file_path": file_path,
                "thread_id": thread_id,
                "message": "AI接口返回空结果",
                "backup_enabled": enable_backup
            }
            
    except Exception as e:
        print(f"[线程{thread_id}] 处理文件时出错 {file_path}: {e}")
        # 如果出现异常且启用了备份，尝试删除可能已创建的备份文件
        if enable_backup and backup_path and os.path.exists(backup_path):
            try:
                os.remove(backup_path)
                print(f"[线程{thread_id}] 已清理备份文件: {backup_path}")
            except:
                pass
        return {
            "success": False,
            "file_path": file_path,
            "thread_id": thread_id,
            "message": f"处理出错: {e}",
            "backup_enabled": enable_backup
        }

def process_files_with_threading(md_files, max_workers=3, enable_backup=True):
    """
    使用多线程处理MD文件
    
    Args:
        md_files (list): MD文件路径列表
        max_workers (int): 最大并发线程数，默认3个
        enable_backup (bool): 是否备份原文件，默认True
    
    Returns:
        list: 处理结果列表
    """
    results = []
    total_files = len(md_files)
    
    backup_status = "启用" if enable_backup else "禁用"
    print(f"\n开始多线程处理 {total_files} 个文件，最大并发数: {max_workers}，备份模式: {backup_status}")
    print("=" * 80)
    
    with ThreadPoolExecutor(max_workers=max_workers) as executor:
        # 提交任务到线程池
        future_to_file = {}
        for i, file_path in enumerate(md_files):
            future = executor.submit(process_single_file, file_path, i + 1, enable_backup)
            future_to_file[future] = file_path
        
        # 获取完成的任务结果
        completed_count = 0
        for future in as_completed(future_to_file):
            completed_count += 1
            result = future.result()
            results.append(result)
            
            status = "✓" if result["success"] else "✗"
            message = result['message']
            if result["success"] and 'original_length' in result and 'new_length' in result:
                change_info = f" (原:{result['original_length']}字符 -> 新:{result['new_length']}字符)"
                if 'content_changed' in result:
                    change_info += f" [内容{'**已更改**' if result['content_changed'] else '未变化'}]"
                # 只在启用备份时显示备份信息
                if result.get('backup_enabled', False) and 'backup_path' in result:
                    change_info += f" [备份: {os.path.basename(result['backup_path'])}]"
                elif not result.get('backup_enabled', True):
                    change_info += f" [无备份]"
                message += change_info
            print(f"{status} [{completed_count}/{total_files}] {message}")
            print(f"    文件: {result['file_path']}")
    
    print("\n" + "=" * 80)
    print("所有文件处理完成！")
    
    # 统计结果
    success_count = sum(1 for r in results if r["success"])
    failed_count = total_files - success_count
    backup_count = sum(1 for r in results if r["success"] and r.get('backup_enabled', False) and 'backup_path' in r)
    
    print(f"处理统计：")
    print(f"  成功: {success_count} 个文件")
    print(f"  失败: {failed_count} 个文件")
    print(f"  备份: {backup_count} 个文件")
    print(f"  备份模式: {backup_status}")
    
    if backup_count > 0:
        print(f"\n备份文件列表：")
        for result in results:
            if result["success"] and result.get('backup_enabled', False) and 'backup_path' in result:
                print(f"  - {result['backup_path']}")
    
    if failed_count > 0:
        print(f"\n失败的文件：")
        for result in results:
            if not result["success"]:
                print(f"  - {result['file_path']}: {result['message']}")
    
    return results

def main():
    """
    主函数：搜索当前运行目录下指定后缀的文件并使用多线程处理
    """
    # 获取当前运行目录
    current_dir = os.getcwd()
    print(f"正在搜索目录: {current_dir}")
    
    try:
        # 搜索所有指定后缀的文件
        found_files = find_files_by_extensions(
            current_dir, 
            CONFIG['FILE_EXTENSIONS'], 
            CONFIG['IGNORE_FILES'], 
            CONFIG['IGNORE_DIRS']
        )
        
        # 打印文件信息
        print_found_files(found_files)
        
        # 额外统计信息
        if found_files:
            total_size = sum(Path(f).stat().st_size for f in found_files if Path(f).exists())
            print(f"\n统计信息:")
            print(f"总文件数: {len(found_files)}")
            print(f"总大小: {total_size} 字节 ({total_size/1024:.2f} KB)")
            
            # 按目录分组统计
            dir_count = {}
            for file_path in found_files:
                dir_name = str(Path(file_path).parent)
                dir_count[dir_name] = dir_count.get(dir_name, 0) + 1
            
            print(f"\n按目录分布:")
            for dir_name, count in dir_count.items():
                print(f"  {dir_name}: {count} 个文件")
            
            # 使用多线程处理所有找到的文件
            print(f"\n开始AI优化处理...")
            print(f"配置信息: 备份模式={CONFIG['ENABLE_BACKUP']}, 最大线程数={CONFIG['MAX_WORKERS']}, 文件后缀={CONFIG['FILE_EXTENSIONS']}")
            print(f"忽略文件: {len(CONFIG['IGNORE_FILES'])} 个模式, 忽略目录: {len(CONFIG['IGNORE_DIRS'])} 个模式")
            results = process_files_with_threading(
                found_files, 
                max_workers=CONFIG['MAX_WORKERS'], 
                enable_backup=CONFIG['ENABLE_BACKUP']
            )
            
        else:
            print(f"\n未找到任何指定后缀的文件，无需处理。")
            print(f"搜索的文件后缀: {CONFIG['FILE_EXTENSIONS']}")
            print(f"忽略文件模式: {CONFIG['IGNORE_FILES']}")
            print(f"忽略目录模式: {CONFIG['IGNORE_DIRS']}")
        
    except Exception as e:
        print(f"搜索过程中出现错误: {e}")

if __name__ == "__main__":
    main()