const fs = require('fs');
const path = require('path');

/**
 * 处理 small 标签的脚本
 * 移除 <small> 标签，将 title 属性值用括号包裹附在元素内容后面
 * 
 * 示例：
 * <small title="陕西">2024-7-19 8:46:21</small>
 * 改为
 * 2024-7-19 8:46:21(陕西)
 */

function processSmallTags(content) {
    // 使用正则表达式匹配 <small title="...">...</small> 格式
    const smallTagRegex = /<small\s+title="([^"]*)"[^>]*>([^<]*)<\/small>/g;
    
    // 替换匹配的内容
    const processedContent = content.replace(smallTagRegex, (match, title, content) => {
        // 将 title 属性值用括号包裹并附加到内容后面
        return `${content}(${title})`;
    });
    
    return processedContent;
}

function main() {
    const inputFile = '时间证明一切，爱到最后全凭良心，权衡利弊后被放弃那个人是幸运的，或许我是幸运的。无缝衔接 断崖式....md';
    const outputFile = 'processed_' + inputFile;
    
    try {
        // 检查输入文件是否存在
        if (!fs.existsSync(inputFile)) {
            console.error(`错误：文件 "${inputFile}" 不存在`);
            return;
        }
        
        // 读取文件内容
        console.log('正在读取文件...');
        const content = fs.readFileSync(inputFile, 'utf8');
        
        // 处理 small 标签
        console.log('正在处理 small 标签...');
        const processedContent = processSmallTags(content);
        
        // 统计处理的标签数量
        const originalMatches = content.match(/<small\s+title="[^"]*"[^>]*>[^<]*<\/small>/g);
        const processedCount = originalMatches ? originalMatches.length : 0;
        
        // 写入处理后的内容
        console.log('正在写入处理后的文件...');
        fs.writeFileSync(outputFile, processedContent, 'utf8');
        
        console.log(`处理完成！`);
        console.log(`- 处理了 ${processedCount} 个 small 标签`);
        console.log(`- 输出文件：${outputFile}`);
        
        // 显示一些示例
        if (originalMatches && originalMatches.length > 0) {
            console.log('\n处理示例：');
            const firstMatch = originalMatches[0];
            const processedExample = processSmallTags(firstMatch);
            console.log(`原文：${firstMatch}`);
            console.log(`处理后：${processedExample}`);
        }
        
    } catch (error) {
        console.error('处理文件时发生错误：', error.message);
    }
}

// 运行主函数
main();
